/**
* \file: libtestbedapp.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: application-framework
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#ifndef LIBTESTBEDAPP_INCLUDE_LIBTESTBEDAPP_H_
#define LIBTESTBEDAPP_INCLUDE_LIBTESTBEDAPP_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "application_fw_constants.h"

#include <stdbool.h>
#include <limits.h>

/**
 * This constant defines a default shutdown timeout which can be passed to libtestbedapp_init()
 * in case no time consuming work is done during shutdown.
 */
#define TBA_DEFAULT_SHUTDOWN_TO			200

/**
 * Definition of an invalid surface id
 */
#define TBA_INVALID_SURFACE_ID			UINT_MAX

/**
 * This enumeration defines error codes which might be returned from the application
 * framework library to the application.
 */
typedef enum tba_error_code_t
{
	/**
	 * No error
	 */
	TBA_RESULT_OK					= 0,

	/**
	 * The application framework ran into issues bringing up its dbus interfaces. See the logs for further details.
	 */
	TBA_RESULT_DBUS_ERROR			= 1,

	/**
	 * The application can not register with the HMI for a certain reason. See the logs for further details.
	 */
	TBA_RESULT_HMI_ERROR			= 2,

	/**
	 * undefined error
	 */
	__TBA_RESULT_UNSET				= 0xFF
} tba_error_code_t;

/**
 * This enumeration defines log level for the application framework library. Log messages are sent to the
 * callback registered with the library according to the loglevel set.
 */
typedef enum tba_loglevel_t
{
	/**
	 * Error, info, and debug logs are sent. This level is meant for debugging.
	 */
	TBA_LOGLEVEL_DEBUG	=	0,

	/**
	 * Error and info logs are sent.
	 */
	TBA_LOGLEVEL_INFO	=	1,

	/**
	 * Only error logs are sent.
	 */
	TBA_LOGLEVEL_ERROR	=	2
} tba_loglevel_t;

/**
 * defines a callback function for receiving log messages.
 * @params level the log level of the message
 * @params message the log message itself
 */
typedef void (*log_event_cb_t)(tba_loglevel_t level, const char *message, void *user_data);

/**
 * The registered callback is called when the state machine changed its state
 *
 * The transition to the state is completely done, when this callback is called.
 * The HMI is signal about the state change after the registered callback function returned.
 * @param new_state the state just entered
 */
typedef void (*state_changed_cb_t)(tba_app_state_t previous_state, tba_app_state_t new_state, void *user_data);

/**
 * called when the library retrieves a surface id from the HMI during registration
 * @param surface_id the surface id to be used by the application to render things
 */
typedef void (*surface_id_received_cb_t)(unsigned int surface_id, void *user_data);

/**
 * Called when the library looses the connection to the HMI to tell the application that
 * the surface_id once provided is not valid any longer.
 * Applications shall stop rendering immediately and remove the surface using the ilm API.
 * @param surface_id the invalidated surface id
 */
typedef void (*surface_id_invalid_cb_t)(unsigned int surface_id, void *user_data);

/**
 * The registerd callback is called by the library when the application shall enter shutdown state.
 *
 * In case the library gets a SHUTDOWN request from node state manager, the library tries to put the application into
 * shutdown state. This callback is called at the beginning of the transition. The shutdown state is not entered before
 * this function is returns. Applications might even delay the transition by returning false. The library stays in
 * active state in this case until libtestbedapp_enter_shutdown_state is finally called.
 *
 * @param request_id the id needs to be passed to libtestbedapp_enter_shutdown_state
 * @return true, if the application is done putting itself into shutdown state after returning this call.
 * Return false in case the transition shall be further delayed. Call libtestbedapp_enter_shutdown_state to finalize
 * the transition later.
 */
typedef bool (*entering_shutdown_cb_t)(int request_id, void *user_data);

/**
 * The registered callback is called by the library when the application shall wake up from shutdown state.
 *
 * In case the library gets a RUNUP request from node state manager, the library tries to reactivate the application.
 * This callback is called at the beginning of the transition. The shutdown state is not left until this function is
 * returned. Applications might even delay the transition by returning false. The library stays in shutdown state in
 * this case until libtestbedapp_enter_active_state is finally called.
 *
 * The callback provides the most recent active state, the application was in before entering shutdown state. Not changing
 * targeted_state will cause the library resuming back into this state. The targeted active state might be changed by
 * the application setting targeted_state respectively.
 *
 * @param recent_active_state contains the state the application was in before going into shutdown state
 * @param targeted_state a point to a state, where the application wants to go into after shutdown
 * @param request_id the id needs to be passed to libtestbedapp_enter_active_state
 * @return true, if the application is done waking up after returning this call. Return false in case the transition
 * shall be further delayed. Call libtestbedapp_enter_active_state to finalize the transition later.
 */
typedef bool (*waking_up_cb_t)(tba_app_state_t recent_active_state, tba_app_state_t *targeted_state,
		int request_id, void *user_data);

/**
 * This structure gets together all callbacks, which might be registered with the application framework library. An
 * instance of this structure can be passed to libtestbedapp_init() to register callbacks.
 */
typedef struct tba_callbacks_t
{
	/**
	 * the callback signaling application state changes
	 */
	const state_changed_cb_t state_changed_cb;

	/**
	 * the callback called before the shutdown state is entered
	 */
	const entering_shutdown_cb_t entering_shutdown_cb;

	/**
	 * the callback called when the shutdown state is left
	 */
	const waking_up_cb_t waking_up_cb;

	/**
	 * this callback is called for every log message that fits to the loglevel set
	 */
	const log_event_cb_t log_event_cb;

	/**
	 * this callback informs the application about a surface id retrieved from HMI
	 */
	const surface_id_received_cb_t surface_id_received_cb;

	/**
	 * this callback tell the application that the previously assigned surface id is not valid any longer
	 */
	const surface_id_invalid_cb_t surface_id_invalid_cb;
} tba_callbacks_t;

/**
 * call to initialize the application framework library
 *
 * @param service_available tell the application library if the service you are providing is already available at the
 * 			time of calling this function. Changes of the service availability might be signal to the library at
 * 			any time using libtestbedapp_service_lost() and libtestbedapp_service_available()
 * @param client_busname a dbus busname which is used to export the NSM client and HMI client interfaces. This busname
 * 			is aquired by the library during initialization. Please ensure to provide the respective dbus configuration
 * 			files to allow your application aquiring this bus
 * @param log_level initialize the application library with a certain loglevel. The loglevel might be changed at any
 * 			time using libtestbedapp_set_loglevel().
 * @param shutdown_timeout Node State Manager waits for the application to enter shutdown state for a certain time.
 * 			Define with this parameter the time in ms, NSM shall wait for you application to enter shutdown state. In
 * 			case no significant work needs to be done before the shutdown can be entered, TBA_DEFAULT_SHUTDOWN_TO can
 * 			be passed.
 * @param callbacks pass a structure with function pointers to register callbacks with the library
 * @param callback_user_data user data could be passed here. The data will be available in the any callback called by the framework
 *
 * @return returns TBA_RESULT_OK in case no errors occurred
 */
tba_error_code_t libtestbedapp_init(bool service_available,const char *client_busname, tba_loglevel_t log_level,
		unsigned int shutdown_timeout, const tba_callbacks_t *callbacks, void *callback_user_data);

/**
 * call to deinitialize the application framework library
 */
void libtestbedapp_deinit(void);

/**
 * call to change the current loglevel.
 * @param the new loglevel
 */
void libtestbedapp_set_loglevel(tba_loglevel_t log_level);

/**
 * Call to get the currently set log level.
 * @return returns the currently set log level
 */
tba_loglevel_t libtestbedapp_get_loglevel(void);

/**
 * Call to get the current application state.
 * @return returns the current application state
 */
tba_app_state_t libtestbedapp_get_state(void);

/**
 * Call to get the most recent application state.
 * @return returns the most recent application state
 */
tba_app_state_t libtestbedapp_get_previous_state(void);

/**
 * Call to get the error message set when state error has been entered.
 *
 * NULL is returned in case the library is not in state TBA_STATE_ERROR.
 *
 * @return returns the currently set error message or NULL in case the library is not in state TBA_STATE_ERROR.
 */
const char *libtestbedapp_get_error_message(void);


/**
 * Call to get the error code set when state error has been entered.
 *
 * TBA_RESULT_OK is returned in case the library is not in state TBA_STATE_ERROR.
 *
 * @return returns the currently set error code or TBA_RESULT_OK in case the library is not in state TBA_STATE_ERROR.
 */
tba_error_code_t libtestbedapp_get_error_code(void);

/**
 * returns the surface id retrieved from the HMI for the application.
 * A valid id is available once state TBA_STATE_HMI_NEGOTIATION has been passed. The value TBA_INVALID_SURFACE_ID
 * is returned in case the library is in state TBA_STATE_INIT, TBA_STATE_HMI_NEGOTIATION, and __TBA_STATE_UNSET.
 *
 * @return returns the surface id retrieved from the HMI for the application or TBA_INVALID_SURFACE_ID in case
 * 		   none has been received so far.
 */
unsigned int libtestbedapp_get_surface_id(void);

/**
 * Call to signal the application framework library that the application wants to go into background state.
 *
 * The HMI is signaled about the state change. Triggered by this change, the HMI will turn off the surface of
 * the application.
 */
void libtestbedapp_send_to_background(void);

/**
 * Call to signal the application framework library that the application wants to go into foreground state.
 *
 * The HMI is signaled about the state change. Triggered by this change, the HMI will change the active surface to
 * the applications one.
 */
void libtestbedapp_send_to_foreground(void);

/**
 * Call to signal the application framework library that the application is not able to provide a service any longer.
 *
 * The application might loose the ability to provide a service during runtime (e.g. a smartphone is removed). The HMI
 * is signaled about the state change. Triggered by this change, the HMI will turn off the surface of the application.
 * Additionally, the button to activate the application again will be disabled.
 */
void libtestbedapp_service_lost(void);

/**
 * Call to signal the application framework library that the application service becomes available.
 *
 * Application might not be able to provide their service all over the time (e.g. a smartphone must be connected).
 * Use this function to tell the library that your application is now able to provide a service. The HMI is signaled
 * about the state change. Triggered by this change, the HMI will enable the button to activate bring the application
 * into foreground.In case you want to trigger the HMI to enter foreground state immediately, please call
 * void libtestbedapp_send_to_foreground() additionally.
 */
void libtestbedapp_service_available(void);

/**
 * Call to send the application into shutdown state.
 *
 * A transition into shutdown state is normally triggered by Node State Manager. Applications might to postpone the
 * transition asynchronously. This function needs to be called to complete the transition to the shutdown state.
 * Pass the request id provided to you in the callback to do so.
 *
 * Applications might decide on their own to enter shutdown state. This can be achieved by calling this function passing
 * -1 as request_id.
 * @param request_id the request id passed to you in the callback triggering the transition to shutdown or -1
 */
void libtestbedapp_enter_shutdown_state(int request_id);

/**
 * Call to transition the application from shutdown back into active state.
 *
 * A transition back into active state is normally triggered by Node State Manager. Applications might postpone the
 * transition asynchronously. This function needs to be called to complete the transition to the active state.
 * Pass the request id provided to you in the callback to do so.
 *
 * Applications might decide on their own to enter active state again. This can be achieved by calling this
 * function passing -1 as request_id. Be careful with this. Normally, a shutdown sequence is ongoing while you are in
 * shutdown state. At the end of the sequence, the head unit is switched off without further notification.
 *
 * The active state to which the application shall transition needs to be passed using the state parameter.
 *
 * @param request_id the request id passed to you in the callback triggering the transition to the active state or -1
 * @param state the active state to be entered
 */
void libtestbedapp_enter_active_state(tba_app_state_t state, int request_id);

#ifdef __cplusplus
}
#endif

#endif /* LIBTESTBEDAPP_INCLUDE_LIBTESTBEDAPP_H_ */
